# PostHog Analytics Overview

This document describes how MiBa Video Manager uses [PostHog](https://posthog.com/) to capture privacy-conscious product analytics that stay within the free tier. It covers credential generation, local development setup, event conventions, and recommended dashboards to understand feature adoption.

## 1. Create Project & Credentials

1. Sign in at [app.posthog.com](https://app.posthog.com/) (or your self-hosted instance) and create a new project dedicated to MiBa Video Manager.
2. Navigate to **Project settings → Project API keys** and copy the **Project API Key** (format `phc_...`). This key is safe to embed in the shipped desktop app.
3. Optional: Generate a **Personal API Key** under **Account settings → Personal API keys**. Keep this secret; it is only used for API automation (e.g., managing dashboards or feature flags) and must **not** ship with the app.
4. Decide where you will store secrets:
   - **Local development**: export environment variables in your shell or an `.env.local` file ignored by git.
   - **CI/build pipelines**: store secrets as CI environment variables or secret manager entries.
   - **Production releases**: only the `POSTHOG_PROJECT_API_KEY` is required at build time. Personal keys remain in CI or ops tooling.

Required environment variables:

| Variable | Required? | Description |
| --- | --- | --- |
| `POSTHOG_PROJECT_API_KEY` | ✅ | Project API key included in the packaged app. |
| `POSTHOG_API_HOST` | ➖ | Custom API host (defaults to `https://app.posthog.com`). |
| `POSTHOG_PERSONAL_API_KEY` | ➖ | Only for CI scripts or tooling. Never bundle with the desktop app. |
| `POSTHOG_EVENTS_CONFIG_URL` | ➖ | Remote JSON that seeds `config/posthog-events.json`. Falls back to bundled defaults if omitted. |
| `POSTHOG_ANALYTICS_DEBUG_ENABLED` | ➖ | Write analytics decision logs to disk (`false` by default). |
| `POSTHOG_ANALYTICS_DEBUG_RETAIN_AFTER_SEND` | ➖ | Keep log entries after successful delivery (`false` by default). |
| `POSTHOG_ANALYTICS_DEBUG_LOG_FILE` | ➖ | Custom filename for the analytics decision log (defaults to `analytics-events.log`). |

## 2. Generate `config/posthog.json`

Run the helper script whenever you need to refresh project credentials locally or in CI:

```bash
POSTHOG_PROJECT_API_KEY="phc_pVeYfqH721630bkBcozSAAak8D1iO8EGg5TgFIfxDOr" \
POSTHOG_API_HOST=https://app.posthog.com \
npm run setup:posthog
```

The command writes `config/posthog.json` containing the project API key and host. The Electron runtime reads this file to initialize PostHog in both renderer and main processes. When `POSTHOG_EVENTS_CONFIG_URL` is provided, the script also seeds or updates `config/posthog-events.json` so that the bundled desktop build knows which events are allowed before the first remote sync completes.

Notes:

- The script never stores personal API keys.
- Commit the generated file for production builds if you want analytics enabled out-of-the-box; omit it if you plan to ask users to opt-in before shipping.
- Keep `config/posthog.example.json` updated with placeholder values so contributors understand the expected shape.
- Set `POSTHOG_EVENTS_CONFIG_URL` to override the remote registry endpoint during CI or when testing staging environments.
- Use the `analyticsDebug` options to control the on-disk decision log (`enabled`, `retainAfterSend`, `logFile`).

## 3. Seed the Event Registry (`config/posthog-events.json`)

The app reads `config/posthog-events.json` to decide which analytics events are allowed, how their payloads are sanitized, and whether they should be dropped before sending to PostHog. The runtime loads this file at start-up, applies any cached rules immediately, then contacts the remote URL defined under `remote.url` (defaults to `https://storage.googleapis.com/cdn.bachmann.xyz/miba-video-manager/posthog-events.json`) to fetch the latest schema.

- **First-launch fallback:** If no cached file exists and the remote endpoint cannot be reached, the app falls back to the bundled schema that only allows `application_start`, `event_schema_download_failed`, and `event_schema_download_success`. These events give us observability without letting legacy builds emit noisy payloads.
- **Remote updates:** After a successful download the config is persisted locally, the in-memory registry is refreshed, and the renderer receives an `analytics.registry.updated` broadcast so UI components can pick up new behaviour without a reload.
- **Manual refresh:** During development you can invoke `ipcRenderer.invoke('analytics.events.refresh')` (or the corresponding main-process handler) to force a foreground refresh. The handler returns whether a new version was applied.
- **Diagnostics logging:** The PostHog config now includes an `analyticsDebug` block (`enabled`, `retainAfterSend`, `logFile`). When enabled, every attempted event is appended to the specified log inside the Electron `userData` directory. Successful deliveries are trimmed automatically unless `retainAfterSend` is set to `true`.

## 4. Local Development Workflow

1. Export `POSTHOG_PROJECT_API_KEY` in your shell (or use a direnv/.env manager).
2. Run `npm run setup:posthog` to sync the config file.
3. Start the app (`npm start`). Analytics will only send events once the user opts in via the settings panel (see shared utilities implementation).
4. To disable analytics locally, delete `config/posthog.json` and relaunch or toggle the opt-in flag in settings.

## 5. Event Taxonomy & Naming

To stay within free-tier limits while capturing actionable insights:

- Use `snake_case` for event names, grouped by surface. Examples: `settings_directory_added`, `viewer_video_played`, `youtube_upload_started`, `youtube_upload_failed`.
- Attach lightweight properties:
  - `source`: `"renderer" | "main" | "transcoder"`
  - `view_mode`: `"gallery" | "player" | "editor"`
  - Counts or boolean flags (e.g., `watched_directory_count`)
- Avoid high-cardinality properties (full file paths, user-entered strings). Instead hash or bucket them where needed.
- Emit one `screen_view` event with `screen_name` whenever the primary UI state changes.
- Track background processes via `posthog-node` using the same event names.

Session replay (recordings) is optional in the free tier. Only enable it after explicit user consent; target short sampling windows (e.g., 10% of sessions) to stay within quotas.

## 6. Dashboard & Reporting Recommendations

Suggested starter assets to create inside PostHog:

- **Feature Usage Dashboard**: charts for top event counts (`viewer_video_played`, `editor_clip_exported`, etc.).
- **User Journey Funnel**: new install → directory added → video played → upload started.
- **Retention Cohort**: users who triggered `viewer_video_played` returning within 7 days.
- **Feature Flag Experiments**: configure flags via PostHog UI (mirrored in the app through the feature flag polling module).

Document any created dashboards in this folder so other contributors can replicate them in additional environments.

## 7. Privacy & Compliance

- Store anonymous analytics identifiers in `settings.json`; reset when the user opts out.
- Provide an in-app toggle with clear wording about the data collected.
- Do not capture raw file paths, video content, or personally identifiable information.
- Honour operating-system network restrictions (analytics should fail open without crashing the app).

## 8. Troubleshooting

- Run the app with `DEBUG=posthog* npm start` to inspect SDK logs.
- If events are missing, verify `config/posthog.json` exists and the `projectApiKey` matches the PostHog project.
- Check `app.posthog.com → Data Warehouse → Live events` to confirm ingestion.
- For CI, rerun `npm run setup:posthog` whenever the project key rotates.
# PostHog Analytics Setup (Free Tier)

This guide explains how to enable privacy-conscious analytics in MiBa Video Manager using the PostHog free plan. The goal is to understand which features users rely on while avoiding paid features or storing sensitive information.

## 1. Create or Reuse a PostHog Project

1. Sign in at [https://app.posthog.com](https://app.posthog.com) (or deploy PostHog OSS if you are self-hosting).
2. Create a new project named something like `MiBa Video Manager`.
3. Inside the project, open **Project settings → Project API key** and copy the value. It starts with `phc_`.
4. (Optional, but recommended for scripting) Navigate to **Profile → Personal API keys**, create a new key, and scope it to the MiBa project. This is only used for developer tooling—never ship it with the app.

## 2. Generate Local Credentials

Set the following environment variables in your shell or `.env.local` file before running the setup script:

```bash
export POSTHOG_PROJECT_API_KEY=phc_example123
# Optional overrides:
export POSTHOG_API_HOST=https://app.posthog.com
export POSTHOG_PERSONAL_API_KEY=phx_your_personal_key   # do not bundle with releases
```

Then create a runtime config file that Electron will read:

```bash
npm run setup:posthog
```

The script writes `config/posthog.json`. For CI/CD you can inject the same environment variables and run the command as part of your build pipeline.

If you ever need to reset the values, delete `config/posthog.json` and re-run the script.

## 3. What the Config Contains

`config/posthog.json` (and the tracked example `config/posthog.example.json`) includes:

- `projectApiKey`: sent from both renderer and main processes when tracking events.
- `apiHost`: defaults to the cloud host; replace it if you self-host.
- `personalApiKey`: optional, used only by developer scripts—never shipped in production builds.
- `featureFlags`: defaults for polling cadence and preload behavior to stay within free-tier limits.
- `capture`: baseline toggles for autocapture and session recording (both disabled by default).

Do **not** commit `config/posthog.json`; add it to your personal `.gitignore` or store it in a secrets manager.

## 4. Measuring Feature Usage (Free Tier)

With the base configuration in place, the app instrumentation (added in subsequent tasks) will publish events that answer:

- Which views users open (`gallery`, `player`, `editor`).
- How often directory scans, search, and filters are used.
- YouTube upload workflow completion, failure, and retry rates.
- Transcoding sessions started vs. completed (main/service processes).

Recommended PostHog artifacts to build once events flow:

- **Dashboards** for top-level engagement (daily active users, feature usage counts).
- **Funnels** comparing video upload journeys (local edit → YouTube publish).
- **Cohorts** tagging power users (e.g., created ≥3 playlists) to inform feature development.

All of the above are included in the PostHog free plan at the time of writing.

## 5. Distribution / CI Notes

- Include `config/posthog.example.json` in the repo so collaborators know the required shape.
- In CI, set the env vars as secure secrets and run `npm run setup:posthog` before packaging the app.
- When shipping production builds, confirm that only the **project API key** is bundled. Personal keys should remain in CI secrets only.

Refer back to this document when adding new product areas to the analytics schema or when rotating credentials.

